<?php

declare(strict_types=1);

require_once __DIR__ . '/sales_service.php';
require_once __DIR__ . '/sales_repo.php';
require_once APP_PATH . '/features/company/company_service.php';
require_once APP_PATH . '/features/masters/masters_repo.php';

function sales_invoices_list_page(): void
{
    $ctx = company_service_require_context();
    $page = max(1, (int) ($_GET['page'] ?? 1));
    $config = config();
    $perPage = $config['pagination']['per_page'] ?? 20;
    $result = sales_repo_list($ctx['company_id'], $ctx['fy_id'], $page, $perPage);
    $totalPages = (int) ceil($result['total'] / $perPage);
    $pageTitle = 'Sales Invoices';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Sales Invoices</h1>
    <a href="?r=sales/invoice/new" class="btn btn-primary">New Invoice</a>
    <table class="data-table">
        <thead>
            <tr><th>Invoice No</th><th>Date</th><th>Customer</th><th>Total</th><th>Status</th><th>Actions</th></tr>
        </thead>
        <tbody>
            <?php foreach ($result['items'] as $si): ?>
            <tr>
                <td><?= e($si['invoice_no']) ?></td>
                <td><?= format_date($si['invoice_date']) ?></td>
                <td><?= e($si['customer_name']) ?></td>
                <td class="text-right"><?= format_money((float) $si['grand_total']) ?></td>
                <td><?= e($si['status']) ?></td>
                <td><a href="?r=sales/invoice/view&id=<?= $si['id'] ?>">View</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php if ($totalPages > 1): ?>
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <a href="?r=sales/invoices&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function sales_invoice_new_page(): void
{
    $ctx = company_service_require_context();
    $customers = masters_repo_ledgers_list($ctx['company_id'], null);
    $customers = array_filter($customers, fn($l) => in_array($l['ledger_type'], ['CUSTOMER', 'GENERAL']));
    $salesLedgers = masters_repo_ledgers_list($ctx['company_id']);
    $salesLedgers = array_filter($salesLedgers, fn($l) => $l['ledger_type'] === 'GENERAL');
    $invoiceNo = sales_repo_next_invoice_no($ctx['company_id'], $ctx['fy_id']);
    $pageTitle = 'New Sales Invoice';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>New Sales Invoice</h1>
    <form method="post" action="?r=sales/invoice/save">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $ctx['company_id'] ?>">
        <input type="hidden" name="fy_id" value="<?= $ctx['fy_id'] ?>">
        <input type="hidden" name="invoice_no" value="<?= e($invoiceNo) ?>">
        <div class="form-group">
            <label for="invoice_date">Date *</label>
            <input type="date" name="invoice_date" required value="<?= date('Y-m-d') ?>">
        </div>
        <div class="form-group">
            <label for="customer_ledger_id">Customer *</label>
            <select name="customer_ledger_id" required>
                <option value="">-- Select --</option>
                <?php foreach ($customers as $c): ?>
                <option value="<?= $c['id'] ?>"><?= e($c['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group">
            <label>Line Items</label>
            <table class="data-table">
                <thead><tr><th>Description</th><th>Qty</th><th>Rate</th><th>Amount</th><th>Ledger (Income)</th></tr></thead>
                <tbody>
                    <?php for ($i = 0; $i < 5; $i++): ?>
                    <tr>
                        <td><input type="text" name="lines[<?= $i ?>][description]" placeholder="Description"></td>
                        <td><input type="number" step="0.01" name="lines[<?= $i ?>][quantity]" value="1" style="width:80px"></td>
                        <td><input type="number" step="0.01" name="lines[<?= $i ?>][rate]" value="0" style="width:100px"></td>
                        <td><input type="number" step="0.01" name="lines[<?= $i ?>][amount]" value="0" style="width:100px" readonly class="line-amt"></td>
                        <td><select name="lines[<?= $i ?>][ledger_id]">
                            <option value="">-- Select --</option>
                            <?php foreach ($salesLedgers as $l): ?>
                            <option value="<?= $l['id'] ?>"><?= e($l['name']) ?></option>
                            <?php endforeach; ?>
                        </select></td>
                    </tr>
                    <?php endfor; ?>
                </tbody>
            </table>
        </div>
        <div class="form-group">
            <label for="discount_amount">Discount</label>
            <input type="number" step="0.01" name="discount_amount" value="0">
        </div>
        <div class="form-group">
            <label for="charges_amount">Charges</label>
            <input type="number" step="0.01" name="charges_amount" value="0">
        </div>
        <div class="form-group">
            <label for="narration">Narration</label>
            <textarea name="narration" rows="2"></textarea>
        </div>
        <button type="submit" class="btn btn-primary">Save Draft</button>
        <a href="?r=sales/invoices" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function sales_invoice_save_action(): void
{
    csrf_verify();
    require_permission('sales.create');
    $ctx = company_service_require_context();

    $lines = $_POST['lines'] ?? [];
    $subtotal = 0;
    foreach ($lines as $l) {
        $qty = (float) ($l['quantity'] ?? 1);
        $rate = (float) ($l['rate'] ?? 0);
        $amt = (float) ($l['amount'] ?? ($qty * $rate));
        $subtotal += $amt;
    }
    $discount = (float) ($_POST['discount_amount'] ?? 0);
    $charges = (float) ($_POST['charges_amount'] ?? 0);
    $grandTotal = $subtotal - $discount + $charges;

    $data = [
        'company_id' => $ctx['company_id'],
        'financial_year_id' => $ctx['fy_id'],
        'invoice_no' => $_POST['invoice_no'] ?? sales_repo_next_invoice_no($ctx['company_id'], $ctx['fy_id']),
        'invoice_date' => $_POST['invoice_date'] ?? date('Y-m-d'),
        'customer_ledger_id' => (int) ($_POST['customer_ledger_id'] ?? 0),
        'subtotal' => $subtotal,
        'discount_amount' => $discount,
        'charges_amount' => $charges,
        'tax_amount' => 0,
        'grand_total' => $grandTotal,
        'narration' => trim($_POST['narration'] ?? ''),
        'created_by' => auth_id(),
    ];

    if ($data['customer_ledger_id'] <= 0) {
        flash_set('error', 'Customer is required.');
        redirect_route('sales/invoice/new');
    }

    $invoiceId = sales_repo_create($data);
    foreach ($lines as $i => $l) {
        $amt = (float) ($l['amount'] ?? 0);
        if ($amt > 0) {
            sales_repo_create_line($invoiceId, [
                'description' => $l['description'] ?? '',
                'quantity' => (float) ($l['quantity'] ?? 1),
                'rate' => (float) ($l['rate'] ?? 0),
                'amount' => $amt,
                'ledger_id' => !empty($l['ledger_id']) ? (int) $l['ledger_id'] : null,
                'sort_order' => $i,
            ]);
        }
    }

    flash_set('success', 'Invoice saved. <a href="?r=sales/invoice/post&id=' . $invoiceId . '">Post now</a>');
    redirect_route('sales/invoice/view', ['id' => $invoiceId]);
}

function sales_invoice_view_page(): void
{
    $ctx = company_service_require_context();
    $id = (int) ($_GET['id'] ?? 0);
    $invoice = sales_repo_find($id);
    if (!$invoice || $invoice['company_id'] != $ctx['company_id']) {
        flash_set('error', 'Invoice not found.');
        redirect_route('sales/invoices');
    }
    $lines = sales_repo_lines($id);
    $pageTitle = 'Invoice ' . $invoice['invoice_no'];

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Sales Invoice: <?= e($invoice['invoice_no']) ?></h1>
    <p><strong>Date:</strong> <?= format_date($invoice['invoice_date']) ?> | <strong>Customer:</strong> <?= e($invoice['customer_name']) ?> | <strong>Status:</strong> <?= e($invoice['status']) ?></p>
    <table class="data-table">
        <thead><tr><th>Description</th><th>Qty</th><th>Rate</th><th>Amount</th></tr></thead>
        <tbody>
            <?php foreach ($lines as $l): ?>
            <tr>
                <td><?= e($l['description'] ?? '') ?></td>
                <td><?= $l['quantity'] ?></td>
                <td class="text-right"><?= format_money((float) $l['rate']) ?></td>
                <td class="text-right"><?= format_money((float) $l['amount']) ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th colspan="3">Subtotal</th><th class="text-right"><?= format_money((float) $invoice['subtotal']) ?></th></tr>
            <tr><th colspan="3">Discount</th><th class="text-right">-<?= format_money((float) $invoice['discount_amount']) ?></th></tr>
            <tr><th colspan="3">Charges</th><th class="text-right"><?= format_money((float) $invoice['charges_amount']) ?></th></tr>
            <tr><th colspan="3">Grand Total</th><th class="text-right"><?= format_money((float) $invoice['grand_total']) ?></th></tr>
        </tfoot>
    </table>
    <?php if ($invoice['status'] === 'DRAFT'): ?>
    <form method="post" action="?r=sales/invoice/post">
        <?= csrf_field() ?>
        <input type="hidden" name="id" value="<?= $invoice['id'] ?>">
        <button type="submit" class="btn btn-primary">Post Invoice</button>
    </form>
    <?php elseif ($invoice['voucher_id']): ?>
    <a href="?r=accounts/voucher/view&id=<?= $invoice['voucher_id'] ?>" class="btn btn-secondary">View Voucher</a>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function sales_invoice_post_action(): void
{
    csrf_verify();
    require_permission('sales.post');
    $id = (int) ($_POST['id'] ?? $_GET['id'] ?? 0);

    try {
        $voucherId = sales_service_post_invoice($id);
        flash_set('success', 'Invoice posted. Voucher #' . $voucherId);
    } catch (Throwable $e) {
        flash_set('error', $e->getMessage());
    }
    redirect_route('sales/invoice/view', ['id' => $id]);
}

function sales_invoice_cancel_action(): void
{
    csrf_verify();
    require_permission('sales.cancel');
    flash_set('info', 'Cancel not implemented yet.');
    redirect_route('sales/invoices');
}
