<?php

declare(strict_types=1);

require_once APP_PATH . '/features/company/company_repo.php';

function reports_repo_trial_balance(int $companyId, int $fyId, ?string $asOfDate = null): array
{
    $pdo = db();
    $fy = company_repo_fy_find($fyId);
    if (!$fy) return [];
    $endDate = $asOfDate ?? $fy['end_date'];

    $stmt = $pdo->prepare("
        SELECT l.id, l.name, l.code, ag.name as group_name, ag.nature,
               COALESCE(lo.opening_debit, 0) as opening_debit,
               COALESCE(lo.opening_credit, 0) as opening_credit,
               COALESCE(SUM(CASE WHEN vl.debit > 0 AND v.voucher_date <= ? THEN vl.debit ELSE 0 END), 0) as period_debit,
               COALESCE(SUM(CASE WHEN vl.credit > 0 AND v.voucher_date <= ? THEN vl.credit ELSE 0 END), 0) as period_credit
        FROM ledgers l
        JOIN account_groups ag ON ag.id = l.group_id
        LEFT JOIN ledger_openings lo ON lo.ledger_id = l.id AND lo.financial_year_id = ?
        LEFT JOIN voucher_lines vl ON vl.ledger_id = l.id
        LEFT JOIN vouchers v ON v.id = vl.voucher_id AND v.company_id = ? AND v.financial_year_id = ? AND v.is_deleted = 0
        WHERE l.company_id = ? AND l.is_active = 1
        GROUP BY l.id, l.name, l.code, ag.name, ag.nature, lo.opening_debit, lo.opening_credit
        HAVING (opening_debit + period_debit - opening_credit - period_credit) != 0
           OR (opening_credit + period_credit - opening_debit - period_debit) != 0
        ORDER BY ag.sort_order, l.name
    ");
    $stmt->execute([$endDate, $endDate, $fyId, $companyId, $fyId, $companyId]);
    return $stmt->fetchAll();
}

function reports_repo_ledger_statement(int $companyId, int $fyId, int $ledgerId, ?string $fromDate = null, ?string $toDate = null): array
{
    $pdo = db();
    $fy = company_repo_fy_find($fyId);
    if (!$fy) return ['opening' => ['debit' => 0, 'credit' => 0], 'lines' => [], 'ledger' => null];

    $from = $fromDate ?? $fy['start_date'];
    $to = $toDate ?? $fy['end_date'];

    $stmt = $pdo->prepare('SELECT l.*, ag.name as group_name FROM ledgers l JOIN account_groups ag ON ag.id = l.group_id WHERE l.id = ? AND l.company_id = ?');
    $stmt->execute([$ledgerId, $companyId]);
    $ledger = $stmt->fetch();
    if (!$ledger) return ['opening' => ['debit' => 0, 'credit' => 0], 'lines' => [], 'ledger' => null];

    $stmt = $pdo->prepare('SELECT opening_debit, opening_credit FROM ledger_openings WHERE ledger_id = ? AND financial_year_id = ?');
    $stmt->execute([$ledgerId, $fyId]);
    $op = $stmt->fetch();
    $openingDr = (float) ($op['opening_debit'] ?? 0);
    $openingCr = (float) ($op['opening_credit'] ?? 0);

    $stmt = $pdo->prepare("
        SELECT v.id as voucher_id, v.voucher_no, v.voucher_date, v.voucher_type, vl.debit, vl.credit, vl.narration
        FROM voucher_lines vl
        JOIN vouchers v ON v.id = vl.voucher_id
        WHERE vl.ledger_id = ? AND vl.company_id = ? AND v.financial_year_id = ? AND v.is_deleted = 0
          AND v.voucher_date >= ? AND v.voucher_date <= ?
        ORDER BY v.voucher_date, v.id
    ");
    $stmt->execute([$ledgerId, $companyId, $fyId, $from, $to]);
    $lines = $stmt->fetchAll();

    return ['opening' => ['debit' => $openingDr, 'credit' => $openingCr], 'lines' => $lines, 'ledger' => $ledger];
}

function reports_repo_pl(int $companyId, int $fyId, ?string $fromDate = null, ?string $toDate = null): array
{
    $pdo = db();
    $fy = company_repo_fy_find($fyId);
    if (!$fy) return [];
    $from = $fromDate ?? $fy['start_date'];
    $to = $toDate ?? $fy['end_date'];

    $stmt = $pdo->prepare("
        SELECT ag.name as group_name, ag.nature, l.name as ledger_name,
               COALESCE(lo.opening_debit, 0) - COALESCE(lo.opening_credit, 0) as opening_net,
               COALESCE(SUM(vl.debit), 0) - COALESCE(SUM(vl.credit), 0) as period_net
        FROM ledgers l
        JOIN account_groups ag ON ag.id = l.group_id
        LEFT JOIN ledger_openings lo ON lo.ledger_id = l.id AND lo.financial_year_id = ?
        LEFT JOIN voucher_lines vl ON vl.ledger_id = l.id
        LEFT JOIN vouchers v ON v.id = vl.voucher_id AND v.company_id = ? AND v.financial_year_id = ? AND v.is_deleted = 0 AND v.voucher_date BETWEEN ? AND ?
        WHERE l.company_id = ? AND l.is_active = 1 AND ag.nature IN ('INCOME', 'EXPENSE')
        GROUP BY l.id, ag.name, ag.nature, l.name, lo.opening_debit, lo.opening_credit
        ORDER BY ag.nature, ag.sort_order, l.name
    ");
    $stmt->execute([$fyId, $companyId, $fyId, $from, $to, $companyId]);
    return $stmt->fetchAll();
}

function reports_repo_bs(int $companyId, int $fyId, ?string $asOfDate = null): array
{
    $pdo = db();
    $fy = company_repo_fy_find($fyId);
    if (!$fy) return [];
    $asOf = $asOfDate ?? $fy['end_date'];

    $stmt = $pdo->prepare("
        SELECT ag.name as group_name, ag.nature, l.name as ledger_name,
               COALESCE(lo.opening_debit, 0) - COALESCE(lo.opening_credit, 0) as opening_net,
               COALESCE(SUM(CASE WHEN v.voucher_date <= ? THEN vl.debit - vl.credit ELSE 0 END), 0) as period_net
        FROM ledgers l
        JOIN account_groups ag ON ag.id = l.group_id
        LEFT JOIN ledger_openings lo ON lo.ledger_id = l.id AND lo.financial_year_id = ?
        LEFT JOIN voucher_lines vl ON vl.ledger_id = l.id
        LEFT JOIN vouchers v ON v.id = vl.voucher_id AND v.company_id = ? AND v.financial_year_id = ? AND v.is_deleted = 0
        WHERE l.company_id = ? AND l.is_active = 1 AND ag.nature IN ('ASSET', 'LIABILITY', 'EQUITY')
        GROUP BY l.id, ag.name, ag.nature, l.name, lo.opening_debit, lo.opening_credit
        ORDER BY ag.nature, ag.sort_order, l.name
    ");
    $stmt->execute([$asOf, $fyId, $companyId, $fyId, $companyId]);
    return $stmt->fetchAll();
}

function reports_repo_daybook(int $companyId, int $fyId, string $date, int $page = 1, int $perPage = 50): array
{
    $pdo = db();
    $offset = ($page - 1) * $perPage;

    $stmt = $pdo->prepare('SELECT COUNT(*) FROM vouchers WHERE company_id = ? AND financial_year_id = ? AND voucher_date = ? AND is_deleted = 0');
    $stmt->execute([$companyId, $fyId, $date]);
    $total = (int) $stmt->fetchColumn();

    $stmt = $pdo->prepare('SELECT * FROM vouchers WHERE company_id = ? AND financial_year_id = ? AND voucher_date = ? AND is_deleted = 0 ORDER BY id LIMIT ? OFFSET ?');
    $stmt->execute([$companyId, $fyId, $date, $perPage, $offset]);
    return ['items' => $stmt->fetchAll(), 'total' => $total];
}
