<?php

declare(strict_types=1);

function purchase_repo_list(int $companyId, int $fyId, int $page = 1, int $perPage = 20): array
{
    $pdo = db();
    $offset = ($page - 1) * $perPage;

    $stmt = $pdo->prepare('SELECT COUNT(*) FROM purchase_bills WHERE company_id = ? AND financial_year_id = ?');
    $stmt->execute([$companyId, $fyId]);
    $total = (int) $stmt->fetchColumn();

    $stmt = $pdo->prepare('
        SELECT pb.*, l.name as supplier_name
        FROM purchase_bills pb
        JOIN ledgers l ON l.id = pb.supplier_ledger_id
        WHERE pb.company_id = ? AND pb.financial_year_id = ?
        ORDER BY pb.bill_date DESC, pb.id DESC
        LIMIT ? OFFSET ?
    ');
    $stmt->execute([$companyId, $fyId, $perPage, $offset]);
    return ['items' => $stmt->fetchAll(), 'total' => $total];
}

function purchase_repo_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT pb.*, l.name as supplier_name FROM purchase_bills pb JOIN ledgers l ON l.id = pb.supplier_ledger_id WHERE pb.id = ?');
    $stmt->execute([$id]);
    return $stmt->fetch() ?: null;
}

function purchase_repo_lines(int $billId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT pbl.*, l.name as ledger_name FROM purchase_bill_lines pbl LEFT JOIN ledgers l ON l.id = pbl.ledger_id WHERE pbl.purchase_bill_id = ? ORDER BY pbl.sort_order');
    $stmt->execute([$billId]);
    return $stmt->fetchAll();
}

function purchase_repo_taxes(int $billId): array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM purchase_bill_taxes WHERE purchase_bill_id = ?');
    $stmt->execute([$billId]);
    return $stmt->fetchAll();
}

function purchase_repo_next_bill_no(int $companyId, int $fyId): string
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT COALESCE(MAX(CAST(SUBSTRING(bill_no, 6) AS UNSIGNED)), 0) + 1 as next FROM purchase_bills WHERE company_id = ? AND financial_year_id = ?');
    $stmt->execute([$companyId, $fyId]);
    $row = $stmt->fetch();
    return 'BILL-' . str_pad((string) ($row['next'] ?? 1), 5, '0', STR_PAD_LEFT);
}

function purchase_repo_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO purchase_bills (company_id, financial_year_id, bill_no, bill_date, supplier_ledger_id, subtotal, discount_amount, charges_amount, tax_amount, grand_total, narration, status, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([
        $data['company_id'],
        $data['financial_year_id'],
        $data['bill_no'],
        $data['bill_date'],
        $data['supplier_ledger_id'],
        $data['subtotal'],
        $data['discount_amount'],
        $data['charges_amount'],
        $data['tax_amount'],
        $data['grand_total'],
        $data['narration'] ?? null,
        $data['status'] ?? 'DRAFT',
        $data['created_by'] ?? null,
    ]);
    return (int) $pdo->lastInsertId();
}

function purchase_repo_create_line(int $billId, array $line): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO purchase_bill_lines (purchase_bill_id, description, quantity, rate, amount, ledger_id, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([
        $billId,
        $line['description'] ?? '',
        $line['quantity'] ?? 1,
        $line['rate'] ?? 0,
        $line['amount'] ?? 0,
        $line['ledger_id'] ?? null,
        $line['sort_order'] ?? 0,
    ]);
}

function purchase_repo_update_voucher(int $billId, int $voucherId): void
{
    $pdo = db();
    $pdo->prepare('UPDATE purchase_bills SET voucher_id = ?, status = ? WHERE id = ?')->execute([$voucherId, 'POSTED', $billId]);
}

function open_refs_repo_create_ap(int $companyId, int $partyLedgerId, string $refModule, int $refId, string $refDate, string $refNumber, float $refAmount): void
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO open_refs (company_id, ref_type, ref_module, ref_id, party_ledger_id, ref_date, ref_number, ref_amount, open_amount, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
    $stmt->execute([$companyId, 'AP_BILL', $refModule, $refId, $partyLedgerId, $refDate, $refNumber, $refAmount, $refAmount, 'OPEN']);
}
