<?php

declare(strict_types=1);

require_once __DIR__ . '/purchase_service.php';
require_once __DIR__ . '/purchase_repo.php';
require_once APP_PATH . '/features/company/company_service.php';
require_once APP_PATH . '/features/masters/masters_repo.php';

function purchase_bills_list_page(): void
{
    $ctx = company_service_require_context();
    $page = max(1, (int) ($_GET['page'] ?? 1));
    $config = config();
    $perPage = $config['pagination']['per_page'] ?? 20;
    $result = purchase_repo_list($ctx['company_id'], $ctx['fy_id'], $page, $perPage);
    $totalPages = (int) ceil($result['total'] / $perPage);
    $pageTitle = 'Purchase Bills';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Purchase Bills</h1>
    <a href="?r=purchase/bill/new" class="btn btn-primary">New Bill</a>
    <table class="data-table">
        <thead>
            <tr><th>Bill No</th><th>Date</th><th>Supplier</th><th>Total</th><th>Status</th><th>Actions</th></tr>
        </thead>
        <tbody>
            <?php foreach ($result['items'] as $pb): ?>
            <tr>
                <td><?= e($pb['bill_no']) ?></td>
                <td><?= format_date($pb['bill_date']) ?></td>
                <td><?= e($pb['supplier_name']) ?></td>
                <td class="text-right"><?= format_money((float) $pb['grand_total']) ?></td>
                <td><?= e($pb['status']) ?></td>
                <td><a href="?r=purchase/bill/view&id=<?= $pb['id'] ?>">View</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php if ($totalPages > 1): ?>
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <a href="?r=purchase/bills&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function purchase_bill_new_page(): void
{
    $ctx = company_service_require_context();
    $suppliers = array_filter(masters_repo_ledgers_list($ctx['company_id']), fn($l) => in_array($l['ledger_type'], ['SUPPLIER', 'GENERAL']));
    $expenseLedgers = array_filter(masters_repo_ledgers_list($ctx['company_id']), fn($l) => $l['ledger_type'] === 'GENERAL');
    $billNo = purchase_repo_next_bill_no($ctx['company_id'], $ctx['fy_id']);
    $pageTitle = 'New Purchase Bill';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>New Purchase Bill</h1>
    <form method="post" action="?r=purchase/bill/save">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $ctx['company_id'] ?>">
        <input type="hidden" name="fy_id" value="<?= $ctx['fy_id'] ?>">
        <input type="hidden" name="bill_no" value="<?= e($billNo) ?>">
        <div class="form-group">
            <label for="bill_date">Date *</label>
            <input type="date" name="bill_date" required value="<?= date('Y-m-d') ?>">
        </div>
        <div class="form-group">
            <label for="supplier_ledger_id">Supplier *</label>
            <select name="supplier_ledger_id" required>
                <option value="">-- Select --</option>
                <?php foreach ($suppliers as $s): ?>
                <option value="<?= $s['id'] ?>"><?= e($s['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group">
            <label>Line Items</label>
            <table class="data-table">
                <thead><tr><th>Description</th><th>Qty</th><th>Rate</th><th>Amount</th><th>Ledger (Expense)</th></tr></thead>
                <tbody>
                    <?php for ($i = 0; $i < 5; $i++): ?>
                    <tr>
                        <td><input type="text" name="lines[<?= $i ?>][description]"></td>
                        <td><input type="number" step="0.01" name="lines[<?= $i ?>][quantity]" value="1" style="width:80px"></td>
                        <td><input type="number" step="0.01" name="lines[<?= $i ?>][rate]" value="0" style="width:100px"></td>
                        <td><input type="number" step="0.01" name="lines[<?= $i ?>][amount]" value="0" style="width:100px"></td>
                        <td><select name="lines[<?= $i ?>][ledger_id]">
                            <option value="">-- Select --</option>
                            <?php foreach ($expenseLedgers as $l): ?>
                            <option value="<?= $l['id'] ?>"><?= e($l['name']) ?></option>
                            <?php endforeach; ?>
                        </select></td>
                    </tr>
                    <?php endfor; ?>
                </tbody>
            </table>
        </div>
        <div class="form-group">
            <label for="discount_amount">Discount</label>
            <input type="number" step="0.01" name="discount_amount" value="0">
        </div>
        <div class="form-group">
            <label for="charges_amount">Charges</label>
            <input type="number" step="0.01" name="charges_amount" value="0">
        </div>
        <div class="form-group">
            <label for="narration">Narration</label>
            <textarea name="narration" rows="2"></textarea>
        </div>
        <button type="submit" class="btn btn-primary">Save Draft</button>
        <a href="?r=purchase/bills" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function purchase_bill_save_action(): void
{
    csrf_verify();
    require_permission('purchase.create');
    $ctx = company_service_require_context();

    $lines = $_POST['lines'] ?? [];
    $subtotal = 0;
    foreach ($lines as $l) {
        $qty = (float) ($l['quantity'] ?? 1);
        $rate = (float) ($l['rate'] ?? 0);
        $amt = (float) ($l['amount'] ?? ($qty * $rate));
        $subtotal += $amt;
    }
    $discount = (float) ($_POST['discount_amount'] ?? 0);
    $charges = (float) ($_POST['charges_amount'] ?? 0);
    $grandTotal = $subtotal - $discount + $charges;

    $data = [
        'company_id' => $ctx['company_id'],
        'financial_year_id' => $ctx['fy_id'],
        'bill_no' => $_POST['bill_no'] ?? purchase_repo_next_bill_no($ctx['company_id'], $ctx['fy_id']),
        'bill_date' => $_POST['bill_date'] ?? date('Y-m-d'),
        'supplier_ledger_id' => (int) ($_POST['supplier_ledger_id'] ?? 0),
        'subtotal' => $subtotal,
        'discount_amount' => $discount,
        'charges_amount' => $charges,
        'tax_amount' => 0,
        'grand_total' => $grandTotal,
        'narration' => trim($_POST['narration'] ?? ''),
        'created_by' => auth_id(),
    ];

    if ($data['supplier_ledger_id'] <= 0) {
        flash_set('error', 'Supplier is required.');
        redirect_route('purchase/bill/new');
    }

    $billId = purchase_repo_create($data);
    foreach ($lines as $i => $l) {
        $amt = (float) ($l['amount'] ?? 0);
        if ($amt > 0) {
            purchase_repo_create_line($billId, [
                'description' => $l['description'] ?? '',
                'quantity' => (float) ($l['quantity'] ?? 1),
                'rate' => (float) ($l['rate'] ?? 0),
                'amount' => $amt,
                'ledger_id' => !empty($l['ledger_id']) ? (int) $l['ledger_id'] : null,
                'sort_order' => $i,
            ]);
        }
    }

    flash_set('success', 'Bill saved.');
    redirect_route('purchase/bill/view', ['id' => $billId]);
}

function purchase_bill_view_page(): void
{
    $ctx = company_service_require_context();
    $id = (int) ($_GET['id'] ?? 0);
    $bill = purchase_repo_find($id);
    if (!$bill || $bill['company_id'] != $ctx['company_id']) {
        flash_set('error', 'Bill not found.');
        redirect_route('purchase/bills');
    }
    $lines = purchase_repo_lines($id);
    $pageTitle = 'Bill ' . $bill['bill_no'];

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Purchase Bill: <?= e($bill['bill_no']) ?></h1>
    <p><strong>Date:</strong> <?= format_date($bill['bill_date']) ?> | <strong>Supplier:</strong> <?= e($bill['supplier_name']) ?> | <strong>Status:</strong> <?= e($bill['status']) ?></p>
    <table class="data-table">
        <thead><tr><th>Description</th><th>Qty</th><th>Rate</th><th>Amount</th></tr></thead>
        <tbody>
            <?php foreach ($lines as $l): ?>
            <tr>
                <td><?= e($l['description'] ?? '') ?></td>
                <td><?= $l['quantity'] ?></td>
                <td class="text-right"><?= format_money((float) $l['rate']) ?></td>
                <td class="text-right"><?= format_money((float) $l['amount']) ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th colspan="3">Subtotal</th><th class="text-right"><?= format_money((float) $bill['subtotal']) ?></th></tr>
            <tr><th colspan="3">Discount</th><th class="text-right">-<?= format_money((float) $bill['discount_amount']) ?></th></tr>
            <tr><th colspan="3">Charges</th><th class="text-right"><?= format_money((float) $bill['charges_amount']) ?></th></tr>
            <tr><th colspan="3">Grand Total</th><th class="text-right"><?= format_money((float) $bill['grand_total']) ?></th></tr>
        </tfoot>
    </table>
    <?php if ($bill['status'] === 'DRAFT'): ?>
    <form method="post" action="?r=purchase/bill/post">
        <?= csrf_field() ?>
        <input type="hidden" name="id" value="<?= $bill['id'] ?>">
        <button type="submit" class="btn btn-primary">Post Bill</button>
    </form>
    <?php elseif ($bill['voucher_id']): ?>
    <a href="?r=accounts/voucher/view&id=<?= $bill['voucher_id'] ?>" class="btn btn-secondary">View Voucher</a>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function purchase_bill_post_action(): void
{
    csrf_verify();
    require_permission('purchase.post');
    $id = (int) ($_POST['id'] ?? $_GET['id'] ?? 0);

    try {
        $voucherId = purchase_service_post_bill($id);
        flash_set('success', 'Bill posted. Voucher #' . $voucherId);
    } catch (Throwable $e) {
        flash_set('error', $e->getMessage());
    }
    redirect_route('purchase/bill/view', ['id' => $id]);
}

function purchase_bill_cancel_action(): void
{
    csrf_verify();
    require_permission('purchase.cancel');
    flash_set('info', 'Cancel not implemented yet.');
    redirect_route('purchase/bills');
}
