<?php

declare(strict_types=1);

require_once __DIR__ . '/masters_service.php';
require_once __DIR__ . '/masters_repo.php';
require_once APP_PATH . '/features/company/company_service.php';

function masters_groups_page(): void
{
    $ctx = company_service_require_context();
    $groups = masters_repo_groups_list($ctx['company_id']);
    $pageTitle = 'Chart of Accounts';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Chart of Accounts - <?= e($ctx['company']['name']) ?></h1>
    <a href="?r=masters/group/create" class="btn btn-primary">Add Group</a>
    <a href="?r=masters/openings" class="btn btn-secondary">Opening Balances</a>
    <table class="data-table">
        <thead>
            <tr><th>Group</th><th>Nature</th><th>Ledgers</th><th>Actions</th></tr>
        </thead>
        <tbody>
            <?php foreach ($groups as $g): ?>
            <tr>
                <td><?= e($g['name']) ?></td>
                <td><?= e($g['nature']) ?></td>
                <td><?= (int) $g['ledger_count'] ?></td>
                <td><a href="?r=masters/ledgers&group_id=<?= $g['id'] ?>">View Ledgers</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function masters_ledgers_page(): void
{
    $ctx = company_service_require_context();
    $groupId = isset($_GET['group_id']) ? (int) $_GET['group_id'] : null;
    $ledgers = masters_repo_ledgers_list($ctx['company_id'], $groupId);
    $groups = masters_repo_groups_list($ctx['company_id']);
    $pageTitle = 'Ledgers';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Ledgers</h1>
    <a href="?r=masters/ledger/create" class="btn btn-primary">Add Ledger</a>
    <form method="get" class="inline-form" style="margin-left:1rem">
        <input type="hidden" name="r" value="masters/ledgers">
        <select name="group_id" onchange="this.form.submit()">
            <option value="">All Groups</option>
            <?php foreach ($groups as $g): ?>
            <option value="<?= $g['id'] ?>" <?= $groupId == $g['id'] ? 'selected' : '' ?>><?= e($g['name']) ?></option>
            <?php endforeach; ?>
        </select>
    </form>
    <table class="data-table">
        <thead>
            <tr><th>Name</th><th>Code</th><th>Type</th><th>Group</th><th>Active</th></tr>
        </thead>
        <tbody>
            <?php foreach ($ledgers as $l): ?>
            <tr>
                <td><?= e($l['name']) ?></td>
                <td><?= e($l['code'] ?? '-') ?></td>
                <td><?= e($l['ledger_type']) ?></td>
                <td><?= e($l['group_name']) ?></td>
                <td><?= $l['is_active'] ? 'Yes' : 'No' ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function masters_group_create_page(): void
{
    $ctx = company_service_require_context();
    $groups = masters_repo_groups_list($ctx['company_id']);
    $pageTitle = 'Add Account Group';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Add Account Group</h1>
    <form method="post" action="?r=masters/group/save">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $ctx['company_id'] ?>">
        <div class="form-group">
            <label for="name">Group Name *</label>
            <input type="text" id="name" name="name" required value="<?= e(old('name')) ?>">
        </div>
        <div class="form-group">
            <label for="nature">Nature *</label>
            <select id="nature" name="nature" required>
                <option value="ASSET">Asset</option>
                <option value="LIABILITY">Liability</option>
                <option value="EQUITY">Equity</option>
                <option value="INCOME">Income</option>
                <option value="EXPENSE">Expense</option>
            </select>
        </div>
        <div class="form-group">
            <label for="parent_id">Parent Group</label>
            <select id="parent_id" name="parent_id">
                <option value="">None</option>
                <?php foreach ($groups as $g): ?>
                <option value="<?= $g['id'] ?>"><?= e($g['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group">
            <label for="sort_order">Sort Order</label>
            <input type="number" id="sort_order" name="sort_order" value="0">
        </div>
        <button type="submit" class="btn btn-primary">Save</button>
        <a href="?r=masters/groups" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function masters_group_save_action(): void
{
    csrf_verify();
    require_permission('masters.create');
    $ctx = company_service_require_context();

    $data = [
        'company_id' => $ctx['company_id'],
        'name' => trim($_POST['name'] ?? ''),
        'nature' => $_POST['nature'] ?? 'ASSET',
        'parent_id' => !empty($_POST['parent_id']) ? (int) $_POST['parent_id'] : null,
        'sort_order' => (int) ($_POST['sort_order'] ?? 0),
    ];

    try {
        masters_service_create_group($data);
        flash_set('success', 'Account group created.');
        redirect_route('masters/groups');
    } catch (Throwable $e) {
        $_SESSION['_old'] = $data;
        flash_set('error', $e->getMessage());
        redirect_route('masters/group/create');
    }
}

function masters_ledger_create_page(): void
{
    $ctx = company_service_require_context();
    $groups = masters_repo_groups_list($ctx['company_id']);
    $pageTitle = 'Add Ledger';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Add Ledger</h1>
    <form method="post" action="?r=masters/ledger/save">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $ctx['company_id'] ?>">
        <div class="form-group">
            <label for="name">Ledger Name *</label>
            <input type="text" id="name" name="name" required value="<?= e(old('name')) ?>">
        </div>
        <div class="form-group">
            <label for="code">Code</label>
            <input type="text" id="code" name="code" value="<?= e(old('code')) ?>">
        </div>
        <div class="form-group">
            <label for="group_id">Account Group *</label>
            <select id="group_id" name="group_id" required>
                <option value="">-- Select --</option>
                <?php foreach ($groups as $g): ?>
                <option value="<?= $g['id'] ?>"><?= e($g['name']) ?> (<?= e($g['nature']) ?>)</option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group">
            <label for="ledger_type">Ledger Type</label>
            <select id="ledger_type" name="ledger_type">
                <option value="GENERAL">General</option>
                <option value="CUSTOMER">Customer</option>
                <option value="SUPPLIER">Supplier</option>
                <option value="BANK">Bank</option>
                <option value="CASH">Cash</option>
                <option value="TAX">Tax</option>
            </select>
        </div>
        <button type="submit" class="btn btn-primary">Save</button>
        <a href="?r=masters/ledgers" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function masters_ledger_save_action(): void
{
    csrf_verify();
    require_permission('masters.create');
    $ctx = company_service_require_context();

    $data = [
        'company_id' => $ctx['company_id'],
        'name' => trim($_POST['name'] ?? ''),
        'code' => trim($_POST['code'] ?? '') ?: null,
        'group_id' => (int) ($_POST['group_id'] ?? 0),
        'ledger_type' => $_POST['ledger_type'] ?? 'GENERAL',
    ];

    try {
        masters_service_create_ledger($data);
        flash_set('success', 'Ledger created.');
        redirect_route('masters/ledgers');
    } catch (Throwable $e) {
        $_SESSION['_old'] = $data;
        flash_set('error', $e->getMessage());
        redirect_route('masters/ledger/create');
    }
}

function masters_openings_page(): void
{
    $ctx = company_service_require_context();
    $openings = masters_repo_openings_list($ctx['company_id'], $ctx['fy_id']);
    $pageTitle = 'Opening Balances';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Opening Balances - <?= e($ctx['fy']['name']) ?></h1>
    <form method="post" action="?r=masters/openings/save">
        <?= csrf_field() ?>
        <input type="hidden" name="fy_id" value="<?= $ctx['fy_id'] ?>">
        <table class="data-table">
            <thead>
                <tr><th>Ledger</th><th>Debit</th><th>Credit</th></tr>
            </thead>
            <tbody>
                <?php foreach ($openings as $o): ?>
                <tr>
                    <td><?= e($o['name']) ?><?= $o['code'] ? ' (' . e($o['code']) . ')' : '' ?></td>
                    <td><input type="number" step="0.01" name="debit[<?= $o['id'] ?>]" value="<?= e($o['opening_debit'] ?? 0) ?>" class="input-amount"></td>
                    <td><input type="number" step="0.01" name="credit[<?= $o['id'] ?>]" value="<?= e($o['opening_credit'] ?? 0) ?>" class="input-amount"></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <button type="submit" class="btn btn-primary">Save Openings</button>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function masters_openings_save_action(): void
{
    csrf_verify();
    require_permission('masters.create');
    $ctx = company_service_require_context();

    $fyId = (int) ($_POST['fy_id'] ?? 0);
    if ($fyId !== $ctx['fy_id']) {
        flash_set('error', 'Invalid FY.');
        redirect_route('masters/openings');
    }

    $debits = $_POST['debit'] ?? [];
    $credits = $_POST['credit'] ?? [];
    $openings = [];
    $ledgers = masters_repo_openings_list($ctx['company_id'], $fyId);
    foreach ($ledgers as $l) {
        $lid = (int) $l['id'];
        $dr = (float) ($debits[$lid] ?? 0);
        $cr = (float) ($credits[$lid] ?? 0);
        if ($dr > 0 || $cr > 0) {
            $openings[] = ['ledger_id' => $lid, 'opening_debit' => $dr, 'opening_credit' => $cr];
        }
    }

    masters_service_save_openings($fyId, $openings);
    flash_set('success', 'Opening balances saved.');
    redirect_route('masters/openings');
}
