<?php

declare(strict_types=1);

require_once __DIR__ . '/company_repo.php';

function company_service_get_context(): array
{
    $companyId = $_SESSION['company_id'] ?? null;
    $fyId = $_SESSION['financial_year_id'] ?? null;

    return [
        'company_id' => $companyId ? (int) $companyId : null,
        'company' => $companyId ? company_repo_find((int) $companyId) : null,
        'fy_id' => $fyId ? (int) $fyId : null,
        'fy' => $fyId ? company_repo_fy_find((int) $fyId) : null,
    ];
}

function company_service_require_context(): array
{
    $ctx = company_service_get_context();
    if (!$ctx['company_id'] || !$ctx['fy_id']) {
        flash_set('error', 'Please select a company and financial year.');
        redirect_route('company/dashboard');
    }
    if (!$ctx['company'] || !$ctx['fy']) {
        flash_set('error', 'Invalid company or financial year selected.');
        redirect_route('company/dashboard');
    }
    return $ctx;
}

function company_service_create(array $data): int
{
    if (empty(trim($data['name'] ?? ''))) {
        throw new InvalidArgumentException('Company name is required.');
    }
    return company_repo_create($data);
}

function company_service_fy_create(array $data): int
{
    if (empty($data['company_id']) || empty($data['name']) || empty($data['start_date']) || empty($data['end_date'])) {
        throw new InvalidArgumentException('All FY fields are required.');
    }
    $start = strtotime($data['start_date']);
    $end = strtotime($data['end_date']);
    if ($start >= $end) {
        throw new InvalidArgumentException('Start date must be before end date.');
    }
    return company_repo_fy_create($data);
}

function company_service_select_fy(int $companyId, int $fyId): void
{
    $company = company_repo_find($companyId);
    $fy = company_repo_fy_find($fyId);
    if (!$company || !$fy || $fy['company_id'] != $companyId) {
        throw new InvalidArgumentException('Invalid company or FY.');
    }
    $_SESSION['company_id'] = $companyId;
    $_SESSION['financial_year_id'] = $fyId;
}
