<?php

declare(strict_types=1);

require_once __DIR__ . '/company_service.php';
require_once __DIR__ . '/company_repo.php';

function company_dashboard_page(): void
{
    $ctx = company_service_get_context();
    $companies = company_repo_list();
    $pageTitle = 'Dashboard';

    require APP_PATH . '/templates/layout_top.php';

    if ($ctx['company_id'] && $ctx['fy_id']):
        ?>
        <div class="dashboard-header">
            <h1>Dashboard</h1>
            <p><strong>Company:</strong> <?= e($ctx['company']['name']) ?> |
               <strong>FY:</strong> <?= e($ctx['fy']['name']) ?> (<?= format_date($ctx['fy']['start_date']) ?> - <?= format_date($ctx['fy']['end_date']) ?>)</p>
            <a href="?r=company/fy/select" class="btn btn-secondary">Change Company/FY</a>
        </div>
        <div class="dashboard-grid">
            <a href="?r=masters/groups" class="card">Chart of Accounts</a>
            <a href="?r=accounts/vouchers" class="card">Vouchers</a>
            <a href="?r=sales/invoices" class="card">Sales</a>
            <a href="?r=purchase/bills" class="card">Purchase</a>
            <a href="?r=reports/trial_balance" class="card">Reports</a>
        </div>
    <?php else:
        $selectedCompanyId = isset($_GET['company_id']) ? (int) $_GET['company_id'] : 0;
        $fys = $selectedCompanyId ? company_repo_fy_list($selectedCompanyId) : [];
        ?>
        <div class="dashboard-select">
            <h1>Select Company & Financial Year</h1>
            <?php if (empty($companies)): ?>
                <p>No companies found. <a href="?r=company/create">Create a company</a> first.</p>
            <?php elseif (!$selectedCompanyId): ?>
                <form method="get" action="?r=company/dashboard">
                    <input type="hidden" name="r" value="company/dashboard">
                    <div class="form-group">
                        <label>Company</label>
                        <select name="company_id" required onchange="this.form.submit()">
                            <option value="">-- Select --</option>
                            <?php foreach ($companies as $c): ?>
                            <option value="<?= $c['id'] ?>"><?= e($c['name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </form>
            <?php elseif (empty($fys)): ?>
                <p>No financial years for this company. <a href="?r=company/fy/create&company_id=<?= $selectedCompanyId ?>">Create a financial year</a> first.</p>
                <a href="?r=company/dashboard" class="btn btn-secondary">Back</a>
            <?php else: ?>
                <form method="post" action="?r=company/fy/select">
                    <?= csrf_field() ?>
                    <input type="hidden" name="company_id" value="<?= $selectedCompanyId ?>">
                    <div class="form-group">
                        <label>Company</label>
                        <p><strong><?= e(company_repo_find($selectedCompanyId)['name'] ?? '') ?></strong> <a href="?r=company/dashboard">Change</a></p>
                    </div>
                    <div class="form-group">
                        <label>Financial Year</label>
                        <select name="fy_id" required>
                            <option value="">-- Select --</option>
                            <?php foreach ($fys as $fy): ?>
                            <option value="<?= $fy['id'] ?>"><?= e($fy['name']) ?> (<?= format_date($fy['start_date']) ?> - <?= format_date($fy['end_date']) ?>)</option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-primary">Continue</button>
                </form>
            <?php endif; ?>
        </div>
    <?php endif;

    require APP_PATH . '/templates/layout_bottom.php';
}

function company_list_page(): void
{
    require_permission('company.view');
    $companies = company_repo_list();
    $pageTitle = 'Companies';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Companies</h1>
    <a href="?r=company/create" class="btn btn-primary">Add Company</a>
    <table class="data-table">
        <thead>
            <tr><th>Name</th><th>Tax ID</th><th>Actions</th></tr>
        </thead>
        <tbody>
            <?php foreach ($companies as $c): ?>
            <tr>
                <td><?= e($c['name']) ?></td>
                <td><?= e($c['tax_id'] ?? '') ?></td>
                <td><a href="?r=company/fy/list&company_id=<?= $c['id'] ?>">Financial Years</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function company_create_page(): void
{
    require_permission('company.create');
    $pageTitle = 'Create Company';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Create Company</h1>
    <form method="post" action="?r=company/save">
        <?= csrf_field() ?>
        <div class="form-group">
            <label for="name">Company Name *</label>
            <input type="text" id="name" name="name" required value="<?= e(old('name')) ?>">
        </div>
        <div class="form-group">
            <label for="address">Address</label>
            <textarea id="address" name="address"><?= e(old('address')) ?></textarea>
        </div>
        <div class="form-group">
            <label for="tax_id">Tax ID</label>
            <input type="text" id="tax_id" name="tax_id" value="<?= e(old('tax_id')) ?>">
        </div>
        <div class="form-group">
            <label for="phone">Phone</label>
            <input type="text" id="phone" name="phone" value="<?= e(old('phone')) ?>">
        </div>
        <div class="form-group">
            <label for="email">Email</label>
            <input type="email" id="email" name="email" value="<?= e(old('email')) ?>">
        </div>
        <button type="submit" class="btn btn-primary">Save</button>
        <a href="?r=company/list" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function company_save_action(): void
{
    csrf_verify();
    require_permission('company.create');

    $data = [
        'name' => trim($_POST['name'] ?? ''),
        'address' => trim($_POST['address'] ?? ''),
        'tax_id' => trim($_POST['tax_id'] ?? ''),
        'phone' => trim($_POST['phone'] ?? ''),
        'email' => trim($_POST['email'] ?? ''),
    ];

    try {
        company_repo_create($data);
        flash_set('success', 'Company created.');
        redirect_route('company/list');
    } catch (Throwable $e) {
        $_SESSION['_old'] = $data;
        flash_set('error', $e->getMessage());
        redirect_route('company/create');
    }
}

function company_fy_list_page(): void
{
    require_permission('company.view');
    $companyId = (int) ($_GET['company_id'] ?? 0);
    $company = $companyId ? company_repo_find($companyId) : null;
    if (!$company) {
        flash_set('error', 'Company not found.');
        redirect_route('company/list');
    }
    $fys = company_repo_fy_list($companyId);
    $pageTitle = 'Financial Years - ' . $company['name'];

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Financial Years - <?= e($company['name']) ?></h1>
    <a href="?r=company/fy/create&company_id=<?= $companyId ?>" class="btn btn-primary">Add FY</a>
    <a href="?r=company/list" class="btn btn-secondary">Back</a>
    <table class="data-table">
        <thead>
            <tr><th>Name</th><th>Start</th><th>End</th><th>Status</th><th>Actions</th></tr>
        </thead>
        <tbody>
            <?php foreach ($fys as $fy): ?>
            <tr>
                <td><?= e($fy['name']) ?></td>
                <td><?= format_date($fy['start_date']) ?></td>
                <td><?= format_date($fy['end_date']) ?></td>
                <td><?= $fy['is_closed'] ? 'Closed' : 'Open' ?></td>
                <td>
                    <a href="?r=company/lock/list&fy_id=<?= $fy['id'] ?>">Locks</a>
                </td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function company_fy_create_page(): void
{
    require_permission('company.create');
    $companyId = (int) ($_GET['company_id'] ?? 0);
    $company = $companyId ? company_repo_find($companyId) : null;
    if (!$company) {
        flash_set('error', 'Company not found.');
        redirect_route('company/list');
    }
    $pageTitle = 'Add Financial Year';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Add Financial Year - <?= e($company['name']) ?></h1>
    <form method="post" action="?r=company/fy/save">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $companyId ?>">
        <div class="form-group">
            <label for="name">Name *</label>
            <input type="text" id="name" name="name" required placeholder="e.g. FY 2024-25" value="<?= e(old('name')) ?>">
        </div>
        <div class="form-group">
            <label for="start_date">Start Date *</label>
            <input type="date" id="start_date" name="start_date" required value="<?= e(old('start_date')) ?>">
        </div>
        <div class="form-group">
            <label for="end_date">End Date *</label>
            <input type="date" id="end_date" name="end_date" required value="<?= e(old('end_date')) ?>">
        </div>
        <button type="submit" class="btn btn-primary">Save</button>
        <a href="?r=company/fy/list&company_id=<?= $companyId ?>" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function company_fy_save_action(): void
{
    csrf_verify();
    require_permission('company.create');

    $data = [
        'company_id' => (int) ($_POST['company_id'] ?? 0),
        'name' => trim($_POST['name'] ?? ''),
        'start_date' => $_POST['start_date'] ?? '',
        'end_date' => $_POST['end_date'] ?? '',
    ];

    try {
        company_repo_fy_create($data);
        flash_set('success', 'Financial year created.');
        redirect_route('company/fy/list', ['company_id' => $data['company_id']]);
    } catch (Throwable $e) {
        $_SESSION['_old'] = $data;
        flash_set('error', $e->getMessage());
        redirect_route('company/fy/create', ['company_id' => $data['company_id']]);
    }
}

function company_fy_select_action(): void
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        csrf_verify();
        require_permission('company.view');
        $companyId = (int) ($_POST['company_id'] ?? 0);
        $fyId = (int) ($_POST['fy_id'] ?? 0);
        try {
            company_service_select_fy($companyId, $fyId);
            flash_set('success', 'Company and FY selected.');
        } catch (Throwable $e) {
            flash_set('error', $e->getMessage());
        }
        redirect_route('company/dashboard');
        return;
    }

    require_permission('company.view');
    $companies = company_repo_list();
    $selectedCompanyId = isset($_GET['company_id']) ? (int) $_GET['company_id'] : 0;
    $fys = $selectedCompanyId ? company_repo_fy_list($selectedCompanyId) : [];
    $pageTitle = 'Select Company & FY';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Select Company & Financial Year</h1>
    <?php if (empty($companies)): ?>
    <p>No companies found. <a href="?r=company/create">Create a company</a> first.</p>
    <?php elseif (!$selectedCompanyId): ?>
    <form method="get" action="?r=company/fy/select">
        <input type="hidden" name="r" value="company/fy/select">
        <div class="form-group">
            <label>Company</label>
            <select name="company_id" required onchange="this.form.submit()">
                <option value="">-- Select --</option>
                <?php foreach ($companies as $c): ?>
                <option value="<?= $c['id'] ?>"><?= e($c['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
    </form>
    <?php elseif (empty($fys)): ?>
    <p>No financial years for this company. <a href="?r=company/fy/create&company_id=<?= $selectedCompanyId ?>">Create a financial year</a> first.</p>
    <a href="?r=company/fy/select" class="btn btn-secondary">Back</a>
    <?php else: ?>
    <form method="post" action="?r=company/fy/select">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $selectedCompanyId ?>">
        <div class="form-group">
            <label>Company</label>
            <p><strong><?= e(company_repo_find($selectedCompanyId)['name'] ?? '') ?></strong> <a href="?r=company/fy/select">Change</a></p>
        </div>
        <div class="form-group">
            <label>Financial Year</label>
            <select name="fy_id" required>
                <option value="">-- Select --</option>
                <?php foreach ($fys as $fy): ?>
                <option value="<?= $fy['id'] ?>"><?= e($fy['name']) ?> (<?= format_date($fy['start_date']) ?> - <?= format_date($fy['end_date']) ?>)</option>
                <?php endforeach; ?>
            </select>
        </div>
        <button type="submit" class="btn btn-primary">Continue</button>
    </form>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function company_lock_list_page(): void
{
    require_permission('company.lock');
    $fyId = (int) ($_GET['fy_id'] ?? 0);
    $fy = $fyId ? company_repo_fy_find($fyId) : null;
    if (!$fy) {
        flash_set('error', 'FY not found.');
        redirect_route('company/dashboard');
    }
    $locks = company_repo_lock_list($fyId);
    $pageTitle = 'Period Locks';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Period Locks - <?= e($fy['name']) ?></h1>
    <form method="post" action="?r=company/lock/save" class="inline-form">
        <?= csrf_field() ?>
        <input type="hidden" name="fy_id" value="<?= $fyId ?>">
        <div class="form-group">
            <label for="lock_date">Lock up to date</label>
            <input type="date" id="lock_date" name="lock_date" required>
        </div>
        <button type="submit" class="btn btn-primary">Add Lock</button>
    </form>
    <a href="?r=company/fy/list&company_id=<?= $fy['company_id'] ?>" class="btn btn-secondary">Back</a>
    <table class="data-table">
        <thead>
            <tr><th>Lock Date</th></tr>
        </thead>
        <tbody>
            <?php foreach ($locks as $l): ?>
            <tr><td><?= format_date($l['lock_date']) ?></td></tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function company_fy_ajax(): void
{
    $companyId = (int) ($_GET['company_id'] ?? 0);
    $fys = $companyId ? company_repo_fy_list($companyId) : [];
    header('Content-Type: application/json');
    echo json_encode($fys);
    exit;
}

function company_lock_save_action(): void
{
    csrf_verify();
    require_permission('company.lock');

    $fyId = (int) ($_POST['fy_id'] ?? 0);
    $lockDate = $_POST['lock_date'] ?? '';

    if (!$fyId || !$lockDate) {
        flash_set('error', 'Invalid data.');
        redirect_route('company/lock/list', ['fy_id' => $fyId]);
    }

    $fy = company_repo_fy_find($fyId);
    if (!$fy || $lockDate < $fy['start_date'] || $lockDate > $fy['end_date']) {
        flash_set('error', 'Lock date must be within FY range.');
        redirect_route('company/lock/list', ['fy_id' => $fyId]);
    }

    company_repo_lock_save($fyId, $lockDate);
    flash_set('success', 'Lock added.');
    redirect_route('company/lock/list', ['fy_id' => $fyId]);
}
