<?php

declare(strict_types=1);

function users_repo_list(): array
{
    $pdo = db();
    $stmt = $pdo->query('SELECT u.*, GROUP_CONCAT(r.name) as roles FROM users u LEFT JOIN user_roles ur ON ur.user_id = u.id LEFT JOIN roles r ON r.id = ur.role_id GROUP BY u.id ORDER BY u.username');
    return $stmt->fetchAll();
}

function users_repo_find(int $id): ?array
{
    $pdo = db();
    $stmt = $pdo->prepare('SELECT * FROM users WHERE id = ?');
    $stmt->execute([$id]);
    return $stmt->fetch() ?: null;
}

function users_repo_find_by_username(string $username, ?int $excludeId = null): ?array
{
    $pdo = db();
    if ($excludeId) {
        $stmt = $pdo->prepare('SELECT * FROM users WHERE username = ? AND id != ?');
        $stmt->execute([$username, $excludeId]);
    } else {
        $stmt = $pdo->prepare('SELECT * FROM users WHERE username = ?');
        $stmt->execute([$username]);
    }
    return $stmt->fetch() ?: null;
}

function users_repo_create(array $data): int
{
    $pdo = db();
    $stmt = $pdo->prepare('INSERT INTO users (username, email, password_hash, name, is_active) VALUES (?, ?, ?, ?, ?)');
    $stmt->execute([
        $data['username'],
        $data['email'] ?? '',
        password_hash($data['password'], PASSWORD_DEFAULT),
        $data['name'] ?? $data['username'],
        $data['is_active'] ?? 1,
    ]);
    return (int) $pdo->lastInsertId();
}

function users_repo_update(int $id, array $data): void
{
    $pdo = db();
    $fields = ['username', 'email', 'name', 'is_active'];
    $sets = [];
    $params = [];
    foreach ($fields as $f) {
        if (array_key_exists($f, $data)) {
            $sets[] = "$f = ?";
            $params[] = $data[$f];
        }
    }
    if (!empty($data['password'])) {
        $sets[] = 'password_hash = ?';
        $params[] = password_hash($data['password'], PASSWORD_DEFAULT);
    }
    if (empty($sets)) return;
    $params[] = $id;
    $sql = 'UPDATE users SET ' . implode(', ', $sets) . ' WHERE id = ?';
    $pdo->prepare($sql)->execute($params);
}

function users_repo_set_roles(int $userId, array $roleIds): void
{
    $pdo = db();
    $pdo->prepare('DELETE FROM user_roles WHERE user_id = ?')->execute([$userId]);
    $stmt = $pdo->prepare('INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)');
    foreach ($roleIds as $rid) {
        $stmt->execute([$userId, $rid]);
    }
}

function users_repo_roles_list(): array
{
    $pdo = db();
    return $pdo->query('SELECT * FROM roles ORDER BY name')->fetchAll();
}
