<?php

declare(strict_types=1);

require_once __DIR__ . '/vouchers_service.php';
require_once __DIR__ . '/vouchers_repo.php';
require_once __DIR__ . '/posting_engine.php';
require_once APP_PATH . '/features/company/company_service.php';
require_once APP_PATH . '/features/masters/masters_repo.php';

function vouchers_list_page(): void
{
    $ctx = company_service_require_context();
    $page = max(1, (int) ($_GET['page'] ?? 1));
    $config = config();
    $perPage = $config['pagination']['per_page'] ?? 20;
    $result = vouchers_repo_list($ctx['company_id'], $ctx['fy_id'], $page, $perPage);
    $totalPages = (int) ceil($result['total'] / $perPage);
    $pageTitle = 'Vouchers';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Vouchers</h1>
    <a href="?r=accounts/voucher/new" class="btn btn-primary">New Voucher</a>
    <table class="data-table">
        <thead>
            <tr><th>No</th><th>Date</th><th>Type</th><th>Narration</th><th>Actions</th></tr>
        </thead>
        <tbody>
            <?php foreach ($result['items'] as $v): ?>
            <tr>
                <td><?= e($v['voucher_no']) ?></td>
                <td><?= format_date($v['voucher_date']) ?></td>
                <td><?= e($v['voucher_type']) ?></td>
                <td><?= e(substr($v['narration'] ?? '', 0, 50)) ?></td>
                <td><a href="?r=accounts/voucher/view&id=<?= $v['id'] ?>">View</a></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php if ($totalPages > 1): ?>
    <div class="pagination">
        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
        <a href="?r=accounts/vouchers&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
        <?php endfor; ?>
    </div>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function voucher_new_page(): void
{
    $ctx = company_service_require_context();
    $ledgers = masters_repo_ledgers_list($ctx['company_id']);
    $pageTitle = 'New Voucher';

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>New Voucher</h1>
    <form method="post" action="?r=accounts/voucher/post" id="voucher_form">
        <?= csrf_field() ?>
        <input type="hidden" name="company_id" value="<?= $ctx['company_id'] ?>">
        <input type="hidden" name="fy_id" value="<?= $ctx['fy_id'] ?>">
        <div class="form-group">
            <label for="voucher_type">Type</label>
            <select name="voucher_type" id="voucher_type">
                <?php foreach (VOUCHER_TYPES as $t): ?>
                <option value="<?= $t ?>"><?= $t ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="form-group">
            <label for="voucher_date">Date *</label>
            <input type="date" name="voucher_date" id="voucher_date" required value="<?= date('Y-m-d') ?>">
        </div>
        <div class="form-group">
            <label for="narration">Narration</label>
            <textarea name="narration" id="narration" rows="2"></textarea>
        </div>
        <table class="data-table" id="voucher_lines">
            <thead>
                <tr><th>Ledger</th><th>Debit</th><th>Credit</th><th>Narration</th><th></th></tr>
            </thead>
            <tbody>
                <tr class="line-row">
                    <td><select name="lines[0][ledger_id]" required>
                        <option value="">-- Select --</option>
                        <?php foreach ($ledgers as $l): ?>
                        <option value="<?= $l['id'] ?>"><?= e($l['name']) ?> (<?= e($l['group_name']) ?>)</option>
                        <?php endforeach; ?>
                    </select></td>
                    <td><input type="number" step="0.01" name="lines[0][debit]" value="0" class="line-debit"></td>
                    <td><input type="number" step="0.01" name="lines[0][credit]" value="0" class="line-credit"></td>
                    <td><input type="text" name="lines[0][narration]"></td>
                    <td></td>
                </tr>
                <tr class="line-row">
                    <td><select name="lines[1][ledger_id]">
                        <option value="">-- Select --</option>
                        <?php foreach ($ledgers as $l): ?>
                        <option value="<?= $l['id'] ?>"><?= e($l['name']) ?></option>
                        <?php endforeach; ?>
                    </select></td>
                    <td><input type="number" step="0.01" name="lines[1][debit]" value="0" class="line-debit"></td>
                    <td><input type="number" step="0.01" name="lines[1][credit]" value="0" class="line-credit"></td>
                    <td><input type="text" name="lines[1][narration]"></td>
                    <td></td>
                </tr>
            </tbody>
        </table>
        <p id="balance_status"></p>
        <button type="submit" class="btn btn-primary">Post</button>
        <a href="?r=accounts/vouchers" class="btn btn-secondary">Cancel</a>
    </form>
    <?php
    require APP_PATH . '/templates/layout_bottom.php';
}

function voucher_post_action(): void
{
    csrf_verify();
    require_permission('accounts.post');
    $ctx = company_service_require_context();

    $data = [
        'company_id' => $ctx['company_id'],
        'fy_id' => $ctx['fy_id'],
        'voucher_type' => $_POST['voucher_type'] ?? 'JOURNAL',
        'voucher_date' => $_POST['voucher_date'] ?? '',
        'narration' => trim($_POST['narration'] ?? ''),
        'lines' => $_POST['lines'] ?? [],
    ];

    try {
        $id = vouchers_service_post($data);
        flash_set('success', 'Voucher posted: ' . $id);
        redirect_route('accounts/voucher/view', ['id' => $id]);
    } catch (Throwable $e) {
        flash_set('error', $e->getMessage());
        $_SESSION['_old'] = $_POST;
        redirect_route('accounts/voucher/new');
    }
}

function voucher_view_page(): void
{
    $ctx = company_service_require_context();
    $id = (int) ($_GET['id'] ?? 0);
    $voucher = vouchers_repo_find($id);
    if (!$voucher || $voucher['company_id'] != $ctx['company_id']) {
        flash_set('error', 'Voucher not found.');
        redirect_route('accounts/vouchers');
    }
    $lines = vouchers_repo_lines($id);
    $pageTitle = 'Voucher ' . $voucher['voucher_no'];

    require APP_PATH . '/templates/layout_top.php';
    ?>
    <h1>Voucher: <?= e($voucher['voucher_no']) ?></h1>
    <p><strong>Date:</strong> <?= format_date($voucher['voucher_date']) ?> | <strong>Type:</strong> <?= e($voucher['voucher_type']) ?></p>
    <p><?= e($voucher['narration'] ?? '') ?></p>
    <?php if ($voucher['ref_module']): ?>
    <p><a href="?r=<?= e($voucher['ref_module']) ?>/view&id=<?= $voucher['ref_id'] ?>">View Source</a></p>
    <?php endif; ?>
    <table class="data-table">
        <thead>
            <tr><th>Ledger</th><th class="text-right">Debit</th><th class="text-right">Credit</th><th>Narration</th></tr>
        </thead>
        <tbody>
            <?php
            $totalDr = 0;
            $totalCr = 0;
            foreach ($lines as $l):
                $totalDr += (float) $l['debit'];
                $totalCr += (float) $l['credit'];
            ?>
            <tr>
                <td><?= e($l['ledger_name']) ?></td>
                <td class="text-right"><?= $l['debit'] > 0 ? format_money((float) $l['debit']) : '' ?></td>
                <td class="text-right"><?= $l['credit'] > 0 ? format_money((float) $l['credit']) : '' ?></td>
                <td><?= e($l['narration'] ?? '') ?></td>
            </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr><th>Total</th><th class="text-right"><?= format_money($totalDr) ?></th><th class="text-right"><?= format_money($totalCr) ?></th><th></th></tr>
        </tfoot>
    </table>
    <?php if (!$voucher['is_deleted']): ?>
    <form method="post" action="?r=accounts/voucher/void" onsubmit="return confirm('Void this voucher?');">
        <?= csrf_field() ?>
        <input type="hidden" name="id" value="<?= $voucher['id'] ?>">
        <input type="text" name="reason" placeholder="Reason for void" required>
        <button type="submit" class="btn btn-secondary">Void</button>
    </form>
    <?php else: ?>
    <p class="alert alert-danger">Voided: <?= e($voucher['void_reason'] ?? '') ?></p>
    <?php endif;
    require APP_PATH . '/templates/layout_bottom.php';
}

function voucher_void_action(): void
{
    csrf_verify();
    require_permission('accounts.void');
    $id = (int) ($_POST['id'] ?? 0);
    $reason = trim($_POST['reason'] ?? '');

    if (!$id || !$reason) {
        flash_set('error', 'Invalid request.');
        redirect_route('accounts/vouchers');
    }

    try {
        void_voucher($id, $reason);
        flash_set('success', 'Voucher voided.');
    } catch (Throwable $e) {
        flash_set('error', $e->getMessage());
    }
    redirect_route('accounts/voucher/view', ['id' => $id]);
}
